/*! \file 
**********************************************************************************
*Title:                         Discretix OMA DRM v2 Toolkit ATP Test source file
*
* Filename:                     ODRM_TLK_ATP_kmac_krek_under_device_key.c  
*
* 
* Created:                      11.12.2007
*
*
* \Author                      Sagit Ben Tsur
*
* \Remarks
*           Copyright (C) 2007 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/
/************* Include Files ***********************************/
#include "DX_VOS_BaseTypes.h"
#include "CRYS.h"
#include "tlk_odrm_types.h"
#include "KMNG_Defs.h" 
#include "ODRM_TLK_ATP_UTIL.h"
#include "KMNG_API.h"
#include "ODRM_TLK_ATP_data.h"
#include "tlk_odrm_api.h"
#include "DX_VOS_Mem.h"
#include "ODRM_TLK_ATP_tests.h"
#include "CRYS_KDF.h"
#include "CRYS_RSA_BUILD.h"
#include "CRYS_RSA_PRIM.h"
#include "CRYS_KMNG.h" 
#include "MW_ATP_UTIL_funcs.h"


/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_TLK_ODRM_Cgenerator 
* 
* Inputs:
*  DxByte_t    * kMacAndkRekBuff     [in] concatanated kMac And kRek buffer
*  DxUint32_t    kMacAndkRekBuffSize [in] concatanated kMac And kRek buffer size
*  DxUint8_t   * pubExponent         [in] Device key public exponent 
*  DxUint16_t    pubExponentSize     [in] Device key public exponent size
*  DxByte_t    * modulus,            [in] Device key modulus
*  DxUint16_t    modulusSize,        [in] Device key modulus Size
*  DxByte_t    *C,                   [out] the result of scheme OMA DRM v2 section 7.2.1
*  DxUint32_t CSize                  [in] C size
*
* return:
*  DxError_t - Function error return                                        
*
* Description:                                                                 
*   Create C using scheme OMA DRM v2 section 7.2.1
*  
* Algorithm: 
* Create C with RSA Pub key Kmac and Krek by using the scheme in OMA DRM v2 section 7.2.1
*******************************************************************************/ 
DxError_t TST_ATP_TLK_ODRM_DEVKCgenerator(DxByte_t    * kMacAndkRekBuff,
                                          DxUint32_t    kMacAndkRekBuffSize,
                                          DxUint8_t   * pubExponent,
                                          DxUint16_t    pubExponentSize,
                                          DxByte_t    * modulus,
                                          DxUint16_t    modulusSize,
                                          DxByte_t    *C,
                                          DxUint32_t CSize  )
{
  
 DxError_t  TST_Error;
 DxByte_t *  KEKkey = TST_gEncryptedDataOut;
 DxUint32_t C2Size = ODRMTLK_TST_C2_SIZE;
 DxUint32_t C1Size = ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES;

 CRYS_RSAUserPubKey_t DevPubKey;
 CRYS_RSAPrimeData_t  PrimeData;


 /*KEK = KDF(I2OSP(Z, mLen), NULL, kekLen)*/
 TST_Error = CRYS_KDF_OMADRM_KeyDerivFunc(TST_gZ,
                                          Z_SIZE,
                                          KEKkey,
                                          ODRMTLK_TST_AES_KEY_SIZE);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"CRYS_KDF_OMADRM_KeyDerivFunc",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

 
  /* C2 = AES-WRAP(KEK, KMAC | KMEK)*/
  TST_Error = CRYS_AES_Wrap(kMacAndkRekBuff,
                            kMacAndkRekBuffSize,
                            KEKkey,
                            CRYS_AES_Key128BitSize,
                            DX_FALSE,
                            TST_gC2Buff,
                            &C2Size);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"CRYS_AES_Wrap",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

  

  /*C1 = I2OSP(RSA.ENCRYPT(PubKeyDevice, Z), mLen) */

  /* Build RSA device public key */
  TST_Error = CRYS_RSA_Build_PubKey(&DevPubKey,
                                    pubExponent,
                                    pubExponentSize,
                                    modulus,
                                    modulusSize);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"CRYS_RSA_Build_PubKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

 
  TST_Error = CRYS_RSA_PRIM_Encrypt(&DevPubKey,
                                    &PrimeData,
                                    TST_gZ,
                                    Z_SIZE,
                                    TST_gC1);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"CRYS_RSA_PRIM_Encrypt",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");
  
 
  /*C = C1 | C2*/
  TST_Error  = DX_VOS_MemCpy(C,CSize,TST_gC1,C1Size); 

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"DX_VOS_MemCpy C1",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");
  
 

  TST_Error  = DX_VOS_MemCpy(C + C1Size,CSize - C1Size ,TST_gC2Buff,C2Size); 

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"DX_VOS_MemCpy C2",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");
  
EXIT_ON_ERROR:
  return TST_Error;
    
}
/*****************************************************************************
* Function Name:
*  TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return
*
* Description:
*    Test TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey 
*  
* Algorithm:
* 1. Get work space size
* 2.  Init the Device key ring  
* 3. Initialize a Key Ring 
* 4. Import RSA private Key  with  KMNG_KEY_USAGE_ODRM and  TLK_ODRM_KEY_RESTRICTION
* 5.  Create C with RSA Pub key Kmac and Krek by using the scheme in OMA DRM v2 section 7.2.1 
* 6. Load  parameters  and Call TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey
* 7. Activate Kmac and Krek from the key ring using the key ids received from the previous operation
* 8.  Verify that these are the same Krek and Kmac as expected
*  8.1 preform HMAC with Kmac on plain text
*  8.2 compare the result with plain text that was Hmac with Kmac in advance
*  8.3 for verifying Krek we will wrap a Cek key with Krek in advance
*       perform  TLK_ODRM_AesUnwrapAndStore with kd as wrapping Key
*       Encrypt plain text  with Cek key and compare with text we encrypted in advance with Cek
*  8.4 load parameters and callTLK_ODRM_AesUnwrapAndStore
*  8.5 Activate  Cek from the key ring using  the key id received from the previous operation
*  8.6 Validate this is the same key we inserted in TLK_ODRM_AesUnwrapAndStore        
*      by Encrypting   plain text with cek from key ring and comparing it with              
*      Plain text we Encrypt in advance with original Cek
*******************************************************************************/ 

DxError_t TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey(void)
{
  TLK_ODRM_KMNGKey_t         kMac;
  KMNG_HMAC_WrappedKey_t     kMac_WrappedKey;
  TLK_ODRM_KMNGKey_t         kRek;
  KMNG_AES_WrappedKey_t      kRek_WrappedKey;
  DxError_t                  TST_Error;
  TLK_ODRM_KMNGKeyRing_t     TLKKeyRing;
  TLK_ODRM_KMNGKeyRing_t     TLKdeviceKeyRing;
  TLK_ODRM_KMNGKey_t         TLKdevPrivKey;
  KMNG_UserSpecificKeyData_t userSpecificData = {0};
  TLK_ODRM_Buffer_t          TLK_C;
  DxByte_t*                  EncryptedPlainBuffWithCek = TST_gEncryptedPlainBuffWithK;
  TLK_ODRM_KMNGKey_t         TLKCekKey;
  KMNG_AES_WrappedKey_t      Cek_WrappedKey;
  TLK_ODRM_Buffer_t          TLKwrappedCekKeyBuff;
   
  ATP_LOG_TST_PRINT((MW_ATP_MSG," ****************************************************** \n"));
  ATP_LOG_TST_PRINT((MW_ATP_MSG," ** TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey Test \n"));
  ATP_LOG_TST_PRINT((MW_ATP_MSG," ****************************************************** \n \n"));


  #ifdef  ODRM_TLK_CLEAR_KEY_RING
      DX_VOS_MemSetZero(TST_gKeyRingBuf,ODRMTLK_TST_MAX_SIZE_KEY_RING_BUF);
      DX_VOS_MemSetZero(TST_gDevKeyRingBuf,ODRMTLK_TST_MAX_SIZE_DEV_KEY_RING_BUF);
  #endif


  /*1. Get work space size */
  TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE,&TST_gWorkspaceSizeInBytes);

  
  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"TLK_ODRM_WorkspaceSizeGet",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");
 
  if(TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
   TST_Error = TST_FAIL;
  else
   TST_Error = TST_PASS;

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"Insufficient Workspace Size ",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");



  /************************************************************************/
  /* 2.Init the Device key ring                                            */
  /************************************************************************/

  TST_gNumOfSymKeys  = 0;
  TST_gNumOfRSAKeys  = 1;
  TST_gNumOfDHKeys   = 0;
  
  /* Get key ring expected buffer size */
  TST_Error = KMNG_GetKeyRingBufferSize( TST_gNumOfSymKeys,  
                                         TST_gNumOfRSAKeys, 
                                         TST_gNumOfDHKeys,  
                                         &TST_gRequiredBufSizeInBytes); 

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_GetKeyRingBufferSize",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");
  

  TST_Error = KMNG_KeyRingInit(TST_gPasswordSize,
                               TST_gPassword,
                               TST_gNumOfSymKeys,  
                               TST_gNumOfRSAKeys,
                               TST_gNumOfDHKeys, 
                               TST_gDevKeyRingBuf);

   TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_KeyRingInit Dev key ring",
                                                 "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                 "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");
  


  /************************************************************************/
  /*3. Init the  key ring                                                 */
  /************************************************************************/
 
  TST_gNumOfSymKeys = 3;
  TST_gNumOfRSAKeys = 0;
  TST_gNumOfDHKeys =  0;
  

  /* Get key ring expected buffer size */
  TST_Error = KMNG_GetKeyRingBufferSize(TST_gNumOfSymKeys,  
                                        TST_gNumOfRSAKeys,
                                        TST_gNumOfDHKeys, 
                                        &TST_gRequiredBufSizeInBytes); 

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_GetKeyRingBufferSize",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

 

  TST_Error = KMNG_KeyRingInit(TST_gPasswordSize,
                               TST_gPassword,
                               TST_gNumOfSymKeys, 
                               TST_gNumOfRSAKeys,
                               TST_gNumOfDHKeys, 
                               TST_gKeyRingBuf);

   TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_KeyRingInit key ring",
                                                 "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                 "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

  

  /************************************************************************/
   /*4. Import RSA private Key                                            */
  /************************************************************************/
   TST_Error = KMNG_ImportRsaUserKey(TST_gPassword,                         /*keyRingPassword_ptr*/
                                     TST_gPasswordSize,                      /*keyRingPasswordLen*/
                                     TST_gPassword,                          /*newKeyPassword_ptr*/
                                     TST_gPasswordSize,                      /*newKeyPasswordLen*/
                                     KMNG_KeyTypeRSAPair,                    /*keyType*/
                                     ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
                                     KMNG_KEY_USAGE_ODRM,                    /*keyUsage*/
                                     TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
                                     userSpecificData,                       /*UserSpecificKeyData*/
                                     TST_gModulus,                           /*modulus_ptr*/
                                     TST_gModulusSize,                       /*modulusSize*/
                                     TST_gPubExponent,                       /*E_ptr*/
                                     TST_gPubExponentSize,                   /*E_Size*/
                                     TST_gPrivExponent,                      /*D_ptr*/
                                     TST_gPrivExponentSize,                  /*D_Size*/
                                     &TLKdevPrivKey.keyId,                   /*keyId_ptr*/
                                     TST_gDevKeyRingBuf);                    /*keyRing_ptr*/


   TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_ImportRsaUserKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");
  
  

  /**********************************************************************************************/
  /* 5. Create C with RSA Pub key Kmac and Krek by using the scheme in OMA DRM v2 section 7.2.1 */
  /*********************************************************************************************/

  TST_Error = TST_ATP_TLK_ODRM_DEVKCgenerator(TST_gKMacAndkRekBuff,
                                             ODRMTLK_TST_DOUBLE_AES_KEY_SIZE,
                                             TST_gPubExponent,
                                             TST_gPubExponentSize,
                                             TST_gModulus,
                                             TST_gModulusSize,
                                             TST_gC,
                                             ODRMTLK_TST_C_SIZE);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"TST_ATP_TLK_ODRM_DEVKCgenerator",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");
    
  
  /*************************************************************************/
  /* 6.load  parameters  and  call TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey */
  /*************************************************************************/
  /************************************************************************/
  /* load Device key ring parameters                                      */
  /************************************************************************/
  TLKdeviceKeyRing.keyKmngRing_ptr = TST_gDevKeyRingBuf;
  TLKdeviceKeyRing.keyKmngRingPassword_ptr = TST_gPassword;
  TLKdeviceKeyRing.keyKmngRingPasswordLenInBytes = TST_gPasswordSize;

  /************************************************************************/
  /* load  key ring parameters                                            */
  /************************************************************************/
  TLKKeyRing.keyKmngRing_ptr = TST_gKeyRingBuf;
  TLKKeyRing.keyKmngRingPassword_ptr = TST_gPassword;
  TLKKeyRing.keyKmngRingPasswordLenInBytes = TST_gPasswordSize;

  /************************************************************************/
  /* load  Device key  parameters                                         */
  /************************************************************************/
  TLKdevPrivKey.keyPassword_ptr = TST_gPassword;
  TLKdevPrivKey.keyPasswordLenInBytes = TST_gPasswordSize;

  /************************************************************************/
  /* load  kMac  password parameters                                      */
  /************************************************************************/
  kMac.keyPassword_ptr = TST_gPassword;
  kMac.keyPasswordLenInBytes = TST_gPasswordSize;

  /************************************************************************/
  /* load  kRek  password parameters                                      */
  /************************************************************************/
  kRek.keyPassword_ptr = TST_gPassword;
  kRek.keyPasswordLenInBytes = TST_gPasswordSize;
  /************************************************************************/
  /* load C Buffer                                                        */
  /************************************************************************/
  TLK_C.buff_ptr = TST_gC;
  TLK_C.buffSizeInBytes = ODRMTLK_TST_C_SIZE;

  TST_Error  = TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey(&TLKdeviceKeyRing,
                                                        &TLKKeyRing,
                                                        &TLKdevPrivKey,
                                                        TLK_C,
                                                        &kMac,
                                                        &kRek,
                                                        TST_gWorkspace,
                                                        ODRMTLK_TST_WORKSPACE_SIZE);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

  
  /*****************************************************************************************************/
  /*7. Activate Kmac and Krek from the key ring using the key ids received from the previous operation */
  /****************************************************************************************************/
  /* Activate Kmac*/
  TST_Error =  KMNG_ActivateHMACKey(TLKKeyRing.keyKmngRingPassword_ptr,
                                    TLKKeyRing.keyKmngRingPasswordLenInBytes,
                                    kMac.keyPassword_ptr, 
                                    kMac.keyPasswordLenInBytes,
                                    kMac.keyId,
                                    TLKKeyRing.keyKmngRing_ptr,
                                    kMac_WrappedKey );

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_ActivateHMACKey  - kMac",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

  TST_Error = KMNG_ActivateAESKey(TLKKeyRing.keyKmngRingPassword_ptr,
                                   TLKKeyRing.keyKmngRingPasswordLenInBytes,
                                   kRek.keyPassword_ptr,
                                   kRek.keyPasswordLenInBytes,
                                   kRek.keyId,
                                   TLKKeyRing.keyKmngRing_ptr,
                                   kRek_WrappedKey);



  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_ActivateAESKey  - Krek",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

 
  /*************************************************************/
  /*8. Verify that these are the same Krek and Kmac as expected  */
  /************************************************************/

  /*8.1 preform HMAC with Kmac on plain text*/
  TST_Error = CRYS_KMNG_HMAC(kMac_WrappedKey,
                             CRYS_HASH_SHA1_mode,
                             TST_gPlainBuff,
                             ODRMTLK_TST_AES_KEY_SIZE,
                             TST_gHmacResultBuff);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"CRYS_KMNG_HMAC  - kMac",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

 
  /* 8.2 compare the result with plain text that was Hmac with Kmac in advance*/
  if(0 != DX_VOS_MemCmp(TST_gHmacPlainBuffWithKmac,TST_gHmacResultBuff,20))
       TST_Error = TST_FAIL; 
  else
    TST_Error = TST_PASS; 

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"DX_VOS_MemCmp - kMac",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");


  /*
  8.3 for verifying Krek we will wrap a Cek key with Krek in advance
  perform  TLK_ODRM_AesUnwrapAndStore with Krek as wrapping Key
  Encrypt plain text  with Cek key and compare with text we encrypted in advance with Cek
  */
  /*Krek */
  /*****************************************************/
  /*8.4 load parameters and callTLK_ODRM_AesUnwrapAndStore */
  /******************************************************/ 

  /************************************************************************/
  /* load wrapped Cek  buff                                               */
  /************************************************************************/
  TLKwrappedCekKeyBuff.buff_ptr            = TST_gWrappedCekWithRek;
  TLKwrappedCekKeyBuff.buffSizeInBytes     = ODRMTLK_TST_WRAPPED_KEY_LENGTH_IN_BYTES;

  /************************************************************************/
  /* load  Cek  pass parameters                                           */
  /************************************************************************/
  TLKCekKey.keyPassword_ptr               = TST_gPassword;
  TLKCekKey.keyPasswordLenInBytes         = TST_gPasswordSize;

  TST_Error = TLK_ODRM_AesUnwrapAndStore(&TLKKeyRing,
                                         &kRek,
                                         &TLKwrappedCekKeyBuff,
                                         &TLKCekKey,
                                         TST_gWorkspace,
                                         ODRMTLK_TST_WORKSPACE_SIZE);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"TLK_ODRM_AesUnwrapAndStore  - kRek",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

 

  /*********************************************************************************************/
  /*8.5  Activate  Cek from the key ring using  the key id received from the previous operation */
  /*********************************************************************************************/


  /*prepares the Cek key for CRYS function */
  TST_Error = KMNG_ActivateAESKey(TST_gPassword,
                                  TST_gPasswordSize,
                                  TLKCekKey.keyPassword_ptr,
                                  TLKCekKey.keyPasswordLenInBytes,
                                  TLKCekKey.keyId,
                                  TST_gKeyRingBuf,
                                  Cek_WrappedKey);


  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"KMNG_ActivateAESKey Cek",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");


  /***************************************************************************************/
  /* 8.6 Validate this is the same key we inserted in TLK_ODRM_AesUnwrapAndStore         */
  /* by Encrypting   plain text with cek from key ring and comparing it with             */
  /* Plain text we Encrypt in advance with original Cek                                  */
  /***************************************************************************************/
  TST_Error = CRYS_KMNG_AES(Cek_WrappedKey,
                            TST_gIvCounter,
                            CRYS_AES_Encrypt,
                            CRYS_AES_ECB_mode,
                            TST_gPlainBuff,
                            ODRMTLK_TST_AES_KEY_SIZE,
                            TST_gEncryptedDataOut);

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"CRYS_KMNG_AES Encrypt - Cek Key",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

  
  if(0 != DX_VOS_MemCmp(EncryptedPlainBuffWithCek,TST_gEncryptedDataOut,ODRMTLK_TST_AES_KEY_SIZE))
    TST_Error = TST_FAIL; 
  else
    TST_Error = TST_PASS; 

  TST_ATP_CHECK_ERROR_RETURN(TST_Error,"DX_VOS_MemCmp - Cekkey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey",
                                                "TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey");

 

  ATP_LOG_TST_PRINT((MW_ATP_MSG," ****************************************************************\n\n"));
  ATP_LOG_TST_PRINT((MW_ATP_MSG," ** TST_ATP_TLK_ODRM_KmacAndKrekDeriveUnderDeviceKey  Test  PASS !!! \n"));     
  ATP_LOG_TST_PRINT((MW_ATP_MSG," **************************************************************** \n \n"));
EXIT_ON_ERROR:
  return TST_PASS;
}
